"use client"

import { useState, useEffect } from "react"
import { Search, UtensilsCrossed, Wine, X } from "lucide-react"

interface MenuNavigationProps {
  onSearch: (query: string) => void
  onFilterChange: (filter: "all" | "food" | "drinks") => void
  activeFilter: "all" | "food" | "drinks"
}

export function MenuNavigation({ onSearch, onFilterChange, activeFilter }: MenuNavigationProps) {
  const [isSticky, setIsSticky] = useState(false)
  const [searchQuery, setSearchQuery] = useState("")
  const [isSearchOpen, setIsSearchOpen] = useState(false)

  useEffect(() => {
    const handleScroll = () => {
      setIsSticky(window.scrollY > 200)
    }
    window.addEventListener("scroll", handleScroll)
    return () => window.removeEventListener("scroll", handleScroll)
  }, [])

  const handleSearchChange = (value: string) => {
    setSearchQuery(value)
    onSearch(value)
  }

  const clearSearch = () => {
    setSearchQuery("")
    onSearch("")
    setIsSearchOpen(false)
  }

  return (
    <nav
      className={`sticky top-0 z-50 transition-all duration-300 ${
        isSticky ? "bg-primary/95 backdrop-blur-md shadow-lg border-b border-accent/20" : "bg-transparent"
      }`}
    >
      <div className="container mx-auto px-4 md:px-6 py-3 md:py-4">
        <div className="flex items-center justify-between gap-4">
          {/* Filter Buttons */}
          <div className="flex items-center gap-1.5 sm:gap-2 md:gap-3">
            <button
              onClick={() => onFilterChange("all")}
              className={`px-2.5 sm:px-3 md:px-4 py-2 rounded-lg text-xs sm:text-sm md:text-base font-medium transition-all duration-300 ${
                activeFilter === "all"
                  ? "bg-accent text-accent-foreground shadow-md"
                  : "bg-muted/50 text-muted-foreground hover:bg-muted"
              }`}
            >
              All
            </button>
            <button
              onClick={() => onFilterChange("food")}
              className={`px-2.5 sm:px-3 md:px-4 py-2 rounded-lg text-xs sm:text-sm md:text-base font-medium transition-all duration-300 flex items-center gap-1.5 sm:gap-2 ${
                activeFilter === "food"
                  ? "bg-accent text-accent-foreground shadow-md"
                  : "bg-muted/50 text-muted-foreground hover:bg-muted"
              }`}
            >
              <UtensilsCrossed className="w-3.5 h-3.5 sm:w-4 sm:h-4" />
              <span className="hidden xs:inline">Food</span>
            </button>
            <button
              onClick={() => onFilterChange("drinks")}
              className={`px-2.5 sm:px-3 md:px-4 py-2 rounded-lg text-xs sm:text-sm md:text-base font-medium transition-all duration-300 flex items-center gap-1.5 sm:gap-2 ${
                activeFilter === "drinks"
                  ? "bg-accent text-accent-foreground shadow-md"
                  : "bg-muted/50 text-muted-foreground hover:bg-muted"
              }`}
            >
              <Wine className="w-3.5 h-3.5 sm:w-4 sm:h-4" />
              <span className="hidden xs:inline">Drinks</span>
            </button>
          </div>

          {/* Search */}
          <div className="flex items-center gap-2">
            {isSearchOpen ? (
              <div className="flex items-center gap-2 bg-muted/50 rounded-lg px-2 sm:px-3 py-2 animate-in fade-in slide-in-from-right-5 duration-300">
                <Search className="w-3.5 h-3.5 sm:w-4 sm:h-4 text-muted-foreground flex-shrink-0" />
                <input
                  type="text"
                  placeholder="Search..."
                  value={searchQuery}
                  onChange={(e) => handleSearchChange(e.target.value)}
                  className="bg-transparent border-none outline-none text-xs sm:text-sm md:text-base w-24 xs:w-32 sm:w-40 md:w-48 text-foreground placeholder:text-muted-foreground"
                  autoFocus
                />
                <button
                  onClick={clearSearch}
                  className="text-muted-foreground hover:text-foreground transition-colors p-1"
                >
                  <X className="w-3.5 h-3.5 sm:w-4 sm:h-4" />
                </button>
              </div>
            ) : (
              <button
                onClick={() => setIsSearchOpen(true)}
                className="p-2 rounded-lg bg-muted/50 text-muted-foreground hover:bg-muted hover:text-foreground transition-all duration-300"
                aria-label="Open search"
              >
                <Search className="w-4 h-4 sm:w-5 sm:h-5" />
              </button>
            )}
          </div>
        </div>
      </div>
    </nav>
  )
}
